﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Domain.Repositories.UnitOfWork;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Microsoft.CodeAnalysis.CSharp;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/insurance-templates")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class InsuranceTemplateController : BaseController
    {
        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly IInsuranceTemplateService insuranceTemplateService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <summary>
        /// The unit of work.
        /// </summary>
        private readonly IUnitOfWork unitOfWork;

        /// <inheritdoc />
        public InsuranceTemplateController(IInsuranceTemplateService insuranceTemplateService, IAuditLogService auditLogServices, IUnitOfWork unitOfWork)
        {
            this.insuranceTemplateService = insuranceTemplateService;
            this.auditLogServices = auditLogServices;
            this.unitOfWork = unitOfWork;
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<InsuranceTemplateModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody]InsuranceTemplateFilterModel model)
        {
            model = (InsuranceTemplateFilterModel)EmptyFilter.Handler(model);
            model.LocationId = model.LocationId > 0 ? model.LocationId : null;
            var insuranceTemplates = await this.insuranceTemplateService.FetchAsync(model);
            return insuranceTemplates == null ? this.ServerError() : this.Success(insuranceTemplates);
        }


        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch-insurance-template")]
        [ProducesResponseType(typeof(List<InsuranceTemplateModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInsuranceTemplateAsync([FromBody] InsuranceTemplateFilterModel model, [FromHeader] LocationHeader header)
        {
            model.LocationId = Convert.ToInt32(header.LocationId);
            var insuranceTemplates = await this.insuranceTemplateService.FetchInsuranceTemplateAsync(model);         
            return Success(insuranceTemplates);
        }


        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] InsuranceTemplateModel model, [FromHeader] LocationHeader header)
        {
            model = (InsuranceTemplateModel)EmptyFilter.Handler(model);          
            var response = await this.insuranceTemplateService.AddAsync(model);

            switch (response)
            {
                case -1:
                    return this.Conflict("Given InsuranceTemplate code has already been exists.");
                case 0:
                    return this.ServerError();
            }
            var templateName = await this.unitOfWork.InsuranceTemplates.FindAsync(m => m.InsuranceTemplateId == response);
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.InsuranceTemplate,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" { model.ModifiedByName } has added Template - {templateName} on { DateTime.UtcNow.AddMinutes(330) }",
                LocationId=Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Template has been added successfully.");
        }

        /// <summary>
        /// The update coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon updated successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] InsuranceTemplateModel model, [FromHeader] LocationHeader header)
        {
            model = (InsuranceTemplateModel)EmptyFilter.Handler(model);
            var response = await this.insuranceTemplateService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Template has already been exists.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.InsuranceTemplate,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" { model.ModifiedByName } has updated template - {model.TemplateName} on { DateTime.UtcNow.AddMinutes(330) }",
                LocationId=Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Template has been updated successfully.");
        }

        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] InsuranceTemplateModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (InsuranceTemplateModel)EmptyFilter.Handler(model);
                var response = await this.insuranceTemplateService.DeleteAsync(model.InsuranceTemplateId);
                switch (response)
                {
                    case -1:
                        return this.Conflict("Unable to Find Template");
                    case -2:
                        return this.Conflict("Template's Bed is allocated, please Deallocate then try");
                    case 0:
                        return this.ServerError();
                }
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.InsuranceTemplate,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $" { model.ModifiedByName } has deleted template - {model.TemplateName} on { DateTime.UtcNow.AddMinutes(330) }",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);

                return this.Success("template has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("Template can't be deleted. Please contact Administrator.");
                }

                return this.ServerError();
            }
        }

        /// <summary>
        /// Changes the status asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpPut]
        [Route("change-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ChangeStatusAsync([FromBody] InsuranceTemplateModel model, [FromHeader] LocationHeader header)
        {
            model = (InsuranceTemplateModel)EmptyFilter.Handler(model);
            var response = await this.insuranceTemplateService.ChangeStatusAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Unable to Find Template");
                case -2:
                    return this.Conflict("Template Name is allocated, please Deallocate then try");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.InsuranceTemplate,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" {model.ModifiedByName} has changed status of  Template - {model.TemplateName}  to {(model.Active ? "Active" : "Deactivate")  } on { DateTime.UtcNow.AddMinutes(330) }",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Template Status has been updated successfully.");
        }

    }
}